<?php

namespace SoundView\Bundle\PasswordListBundle\Form\Type;

use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolverInterface;


use SoundView\Bundle\PasswordListBundle\Entity\Password;
use Oro\Bundle\SecurityBundle\Encoder\Mcrypt;

class PasswordType extends AbstractType
{
    /** @var Mcrypt */
    protected $encryptor;

    public function __construct(Mcrypt $encryptor)
    {
        $this->encryptor = $encryptor;
    }

        /**
     * @param FormBuilderInterface $builder
     * @param array $options
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $encryptor = $this->encryptor;

        // pre-populate password, imap origin change
        $builder->addEventListener(
            FormEvents::PRE_SUBMIT,
            function (FormEvent $event) use ($encryptor) {
                $data = (array) $event->getData();
                /** @var Password|null $entity */
                $entity = $event->getForm()->getData();

                $filtered = array_filter(
                    $data,
                    function ($item) {
                        return !empty($item);
                    }
                );

                if (!empty($filtered)) {
                    $oldPassword = $event->getForm()->get('password')->getData();
                    if (empty($data['password']) && $oldPassword) {
                        // populate old password
                        $data['password'] = $oldPassword;
                    } else {
                        $data['password'] = $encryptor->encryptData($data['password']);
                    }


                    if ($entity instanceof Password
                        && ($this->encryptor->decryptData($oldPassword) != $this->encryptor->decryptData($data['password']))
                    ) {
                        // in case when critical fields were changed new entity should be created
                        $data['lastPassword'] = $this->encryptor->decryptData($oldPassword);
                    }
                    $event->setData($data);

                } elseif ($entity instanceof Password) {
                    $event->getForm()->setData(null);
                }
            }
        );


        $builder
            ->add('domain', 'text', [ 'label' => 'Domain', 'required' => true])
            ->add('user', 'text', ['label' => 'User Name', 'required' => true])
            ->add('password', 'password', ['label' => 'Password', 'required' => true])
            ->add('lastPassword', 'hidden')
            ->add('public', 'checkbox', ['label' => 'Public', 'required' => true])
        ;
    }
    
    /**
     * @param OptionsResolverInterface $resolver
     */
    public function setDefaultOptions(OptionsResolverInterface $resolver)
    {
        $resolver->setDefaults(
            array(
                'data_class'           => 'SoundView\Bundle\PasswordListBundle\Entity\Password',
                'intention'            => 'password',
                'extra_fields_message' => 'This form should not contain extra fields: "{{ extra_fields }}"',
                'cascade_validation'   => true,
            )
        );
    }

    /**
     * @return string
     */
    public function getName()
    {
        return 'soundview_password_list';
    }
}
