<?php

namespace SoundView\Bundle\SaleBundle\Controller\Api\Rest;

use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;

use Nelmio\ApiDocBundle\Annotation\ApiDoc;

use FOS\RestBundle\Controller\Annotations\NamePrefix;
use FOS\RestBundle\Controller\Annotations\RouteResource;
use FOS\RestBundle\Controller\Annotations\QueryParam;
use FOS\RestBundle\Routing\ClassResourceInterface;
use FOS\Rest\Util\Codes;


use Oro\Bundle\SecurityBundle\Annotation\Acl;
use Oro\Bundle\SecurityBundle\Annotation\AclAncestor;
use Oro\Bundle\SoapBundle\Entity\Manager\ApiEntityManager;
use Oro\Bundle\SoapBundle\Form\Handler\ApiFormHandler;
use Oro\Bundle\SoapBundle\Controller\Api\Rest\RestController;

use SoundView\Bundle\SaleBundle\Entity\Sale;
use SoundView\Bundle\SaleBundle\Entity\Payment;
use SoundView\Bundle\SaleBundle\Entity\PurchaseOrder;

/**
 * @RouteResource("payment")
 * @NamePrefix("svwd_api_")
 */
class PaymentController extends RestController implements ClassResourceInterface
{

    /**
     * REST GET payment
     *
     * @param string $saleId
     * @param string $paymentId
     *
     * @ApiDoc(
     *      description="Get sale po",
     *      resource=true
     * )
     * @AclAncestor("svwd_sale_view")
     * @return Response
     */
    public function getAction($saleId, $paymentId)
    {
        /** @var Sale $sale */
        $sale = $this->getSaleManager()->find($saleId);

        /** @var Payment $payment */
        $payment = $this->getManager()->find($paymentId);

        $addressData = null;
        if ($payment && $sale->getPayments()->contains($payment)) {
            $addressData = $this->getPreparedItem($payment);
        }
        $responseData = $addressData ? json_encode($addressData) : '';
        return new Response($responseData, $payment ? Codes::HTTP_OK : Codes::HTTP_NOT_FOUND);
    }
	
    /**
     * REST GET list
     *
     * @ApiDoc(
     *      description="Get all payment items",
     *      resource=true
     * )
     * @AclAncestor("svwd_sale_view")
     * @param int $saleId
     *
     * @return JsonResponse
     */
    public function cgetAction($saleId)
    {
        /** @var Sale $sale */
        $sale = $this->getSaleManager()->find($saleId);
        $result  = [];

        if (!empty($sale)) {
            $items = $sale->getPayments();

            foreach ($items as $item) {
                $result[] = $this->getPreparedItem($item);
            }
        }

        return new JsonResponse(
            $result,
            empty($sale) ? Codes::HTTP_NOT_FOUND : Codes::HTTP_OK
        );
    }


    /**
     * REST PUT
     *
     * @param int $id Payment item id
     *
     * @ApiDoc(
     *      description="Update po",
     *      resource=true
     * )
     * @AclAncestor("svwd_sale_update")
     * @return Response
     */
    public function putAction($id)
    {
        return $this->handleUpdateRequest($id);
    }

    /**
     * Create new po
     *
     * @ApiDoc(
     *      description="Create new po",
     *      resource=true
     * )
     * @AclAncestor("svwd_sale_create")
     */
    public function postAction()
    {
        return $this->handleCreateRequest();
    }

    /**
     * REST DELETE
     *
     * @param int $id
     *
     * @ApiDoc(
     *      description="Delete payment",
     *      resource=true
     * )
     * @Acl(
     *      id="svwd_payment_delete",
     *      type="entity",
     *      permission="DELETE",
     *      class="SoundViewSaleBundle:Payment"
     * )
     * @return Response
     */
    public function deleteAction($id)
    {
        return $this->handleDeleteRequest($id);
    }

    /**
     * Get entity Manager
     *
     * @return ApiEntityManager
     */
    public function getManager()
    {
        return $this->get('svwd_sale.payment.manager.api');
    }

    public function getSaleManager()
    {
        return $this->get('svwd_sale.sale.manager.api');
    }

    /**
     * @return FormInterface
     */
    public function getForm()
    {
        return $this->get('svwd_sale.payment.form.api');
    }

    /**
     * @return ApiFormHandler
     */
    public function getFormHandler()
    {
        return $this->get('svwd_sale.payment.form.handler.api');
    }

    /**
     * {@inheritdoc}
     */
    protected function transformEntityField($field, &$value)
    {
        switch ($field) {
            case 'owner':
            default:
                parent::transformEntityField($field, $value);
        }
    }

     protected function getPreparedItem($entity, $resultFields = [])
    {


        $result                = parent::getPreparedItem($entity);

        unset($result['owner']);

        return $result;
    }

    /**
     * {@inheritDoc}
     */
    protected function fixFormData(array &$data, $entity)
    {
        parent::fixFormData($data, $entity);

        unset($data['id']);
        unset($data['createdAt']);
        unset($data['updatedAt']);

        return true;
    }
}
