/*global define*/
define([
    'underscore',
    'backbone',
    'orotranslation/js/translator',
    'oroui/js/mediator', 'oroui/js/messenger', 'oroui/js/error',
    'oro/dialog-widget',
    'svwdsale/js/po/view',
    'svwdsale/js/po/collection'
], function (
    _,
    Backbone,
    __,
    mediator,
    messenger,
    error,
    DialogWidget,
    AddressView,
    AddressCollection
) {
    'use strict';

    var $ = Backbone.$;

    /**
     * @export  oroaddress/js/address-book
     * @class   oroaddress.AddressBook
     * @extends Backbone.View
     */
    return Backbone.View.extend({
        options: {
            'template': null,
            'addressListUrl': null,
            'addressCreateUrl': null,
            'addressUpdateUrl': null,
            'addressViewUrl': null,
            'addressReceiveUrl': null
        },
        noDataMessage: 'No Purchase Orders',
        attributes: {
            'class': ''
        },

        initialize: function (options) {
            this.options = _.defaults(options || {}, this.options);
            this.options.collection = this.options.collection || new AddressCollection();
            this.options.collection.url = this._getUrl('addressListUrl');

            this.listenTo(this.getCollection(), 'activeChange', this.activateAddress);
            this.listenTo(this.getCollection(), 'add', this.addAddress);
            this.listenTo(this.getCollection(), 'reset', this.addAll);
            this.listenTo(this.getCollection(), 'remove', this.onAddressRemove);

            this._initMainContainers();

        },

        _initMainContainers: function() {
            this.$noDataContainer   = $('<div class="no-data"><span>' + this.noDataMessage + '</span></div>');
            this.$adressesContainer = $('<div class="list-group po-list"/>');

            if (!this.$el.find('.list-group.po-list').length) {
                this.$el.append(this.$adressesContainer);
            }


            if (!this.$el.find('.no-data').length) {
                this.$el.append(this.$noDataContainer);
            }
        },

        _getUrl: function (optionsKey) {
            if (_.isFunction(this.options[optionsKey])) {
                return this.options[optionsKey].apply(this, Array.prototype.slice.call(arguments, 1));
            }
            return this.options[optionsKey];
        },

        getCollection: function () {
            return this.options.collection;
        },

        onAddressRemove: function () {
            if (!this.getCollection().where({active: true}).length) {
                var primaryAddress = this.getCollection().where({primary: true});
                if (primaryAddress.length) {
                    primaryAddress[0].set('active', true);
                } else if (this.getCollection().length) {
                    this._activateFirstAddress();
                }
            }
        },

        _activateFirstAddress: function () {
            this.getCollection().at(0).set('active', true);
        },

        addAll: function (items) {
            this.$adressesContainer.empty();
            if(items.length > 0){
                this._hideEmptyMessage();
                items.each(function (item) {
                    this.addAddress(item);
                }, this);
            } else {
                this._showEmptyMessage();
            }
            if (items.length === 1) {
                this._activateFirstAddress();
            } else {
                this._activatePreviousAddress();
            }
        },

        _hideEmptyMessage: function(){
            this.$noDataContainer.hide();
            this.$adressesContainer.show();
        },

        _showEmptyMessage: function () {
            this.$noDataContainer.show();
            this.$adressesContainer.hide();
        },

        _activatePreviousAddress: function () {
            if (this.activeAddress !== undefined) {
                var previouslyActive = this.getCollection().where({id: this.activeAddress.get('id')});
                if (previouslyActive.length) {
                    previouslyActive[0].set('active', true);
                }
            }
        },

        addAddress: function (address) {
            if (!this.$el.find('#po-book-' + address.id).length) {
                var addressView = new AddressView({
                    model: address
                });
                addressView.on('edit', _.bind(this.editAddress, this));
                addressView.on('view', _.bind(this.viewAddress, this));
                addressView.on('receive', _.bind(this.receiveAddress, this));
                this.$adressesContainer.append(addressView.render().$el);
            }
        },
        receiveAddress: function(addressView, address) {
            $.ajax({
                url: this._getUrl('addressReceiveUrl', address),
                context: this,
                dataType: 'json',
                error: this._onAjaxError,
                success: this._onAjaxSuccess
            });
        },
        viewAddress: function (addressView, address) {
            this._openAddressViewDialog(address.get('poNumber'), this._getUrl('addressViewUrl', address));
        },
        editAddress: function (addressView, address) {
            this._openAddressEditForm('Edit PO', this._getUrl('addressUpdateUrl', address));
        },

        createAddress: function () {
            this._openAddressEditForm('Add PO', this._getUrl('addressCreateUrl'));
        },
        _openAddressViewDialog: function (title, url) {
            if (!this.addressEditDialog) {
                this.addressEditDialog = new DialogWidget({
                    'url': url,
                    'title': title,
                    'regionEnabled': false,
                    'incrementalPosition': false,
                    'dialogOptions': {
                        'modal': true,
                        'resizable': true,
                        'width': 1000,
                        'autoResize': true,
                        'close': _.bind(function () {
                            delete this.addressEditDialog;
                        }, this)
                    }
                });
                this.addressEditDialog.render();
                mediator.on(
                    "page:request",
                    _.bind(function () {
                        if (this.addressEditDialog) {
                            this.addressEditDialog.remove();
                        }
                    }, this)
                );
            }
        },

        _openAddressEditForm: function (title, url) {
            if (!this.addressEditDialog) {
                this.addressEditDialog = new DialogWidget({
                    'url': url,
                    'title': title,
                    'regionEnabled': false,
                    'incrementalPosition': false,
                    'dialogOptions': {
                        'modal': true,
                        'resizable': true,
                        'width': 1000,
                        'autoResize': true,
                        'close': _.bind(function () {
                            delete this.addressEditDialog;
                        }, this)
                    }
                });
                this.addressEditDialog.render();
                mediator.on(
                    "page:request",
                    _.bind(function () {
                        if (this.addressEditDialog) {
                            this.addressEditDialog.remove();
                        }
                    }, this)
                );
                this.addressEditDialog.on('formSave', _.bind(function () {
                    this.addressEditDialog.remove();
                    messenger.notificationFlashMessage('success', 'PO saved');
                    this.reloadAddresses();
                }, this));
            }
        },
        _onAjaxError: function (jqXHR) {
            error.handle({}, jqXHR, {enforce: true});
        },

        _onAjaxSuccess: function (data) {
            messenger.notificationFlashMessage('success', 'PO received');
            this.reloadAddresses();
        },

        _showAjaxSuccessMessage: function (data) {
            var defaultMessage = data.successful ? this.messages.success : this.messages.error,
                type = data.successful ? 'success' : 'error',
                message = data.message || __(defaultMessage);
            if (message) {
                mediator.execute('showFlashMessage', type, message);
            }
        },

        reloadAddresses: function () {
            this.getCollection().fetch({reset: true});
        },

        activateAddress: function (address) {
            if (!address.get('primary')) {
                this.activeAddress = address;
            }
        }
    });
});
